package net.wasamon.blokus.javarock;

import net.wasamon.javarock.rt.unsynthesizable;


public class BlokusBoard {

	public static final int NONE = 0;
	public static final int BLUE = 1;
	public static final int ORANGE = 2;

	public static final int COLS = 14;
	public static final int ROWS = 14;

	private final int[] map = new int[256]; // more than COLS * ROWS (= 196)
	private final byte[] sym = new byte[256]; // more than COLS * ROWS (= 196)
	
	private final BlokusPatternTable table = new BlokusPatternTable();
	private final BlokusPatternLengthTable lenTable = new BlokusPatternLengthTable();
	private final BlokusPatternSymTable symTable = new BlokusPatternSymTable();

	public void init() {
		for (int i = 0; i < map.length; i++) {
			map[i] = NONE;
			sym[i] = NONE;
		}
	}

	private void put(int c, int x, int y, byte s) {
		map[(y << 4) + x] = c; // y * COLS
		sym[(y << 4) + x] = s; // y * COLS
	}

	public int get(int x, int y) {
		return map[(y << 4) + x]; // y * COLS
	}
	
	public byte getSym(int x, int y){
		return sym[(y << 4) + x]; // y * COLS
	}
	
	private final int[] check_x = new int[5];
	private final int[] check_y = new int[5];
	private int check_len = 0;
	
	private boolean isInsideAndEmpty() {
		int x, y;
		for(int i = 0; i < check_len; i++){
			x = check_x[i];
			y = check_y[i];
			if(x < 0 || y < 0 || x >= COLS || y >= ROWS){
				return false;
			}else if(map[(y<<4) + x] != NONE){
				return false;
			}
		}
		return true;
	}

	private boolean isDuplicated(int c, int dx, int dy) {
		int x, y;
		for (int i = 0; i < check_len; i++) {
			x = check_x[i] + dx;
			y = check_y[i] + dy;
			if(x >= 0 && y >= 0 && x < COLS && y < ROWS){
				if (map[(y<<4) + x] == c) {
					return true;
				}
			}
		}
		return false;
	}
	
	private boolean isEdgeSharing(int c){
		boolean flag0, flag1, flag2, flag3;
		// NORTH
		flag0 = isDuplicated(c, 0, -1);
		// WEST
		flag1 = isDuplicated(c, -1, 0);
		// EAST
		flag2 = isDuplicated(c, 1, 0);
		// SOUTH
		flag3 = isDuplicated(c, 0, 1);
		if(flag0 == true || flag1 == true || flag2 == true || flag3 == true){
			return true;
		}else{
			return false;
		}
	}

	private boolean isVertexSharing(int c){
		boolean flag0, flag1, flag2, flag3;
		// NORTH-WEST
		flag0 = isDuplicated(c, -1, -1);
		// NORTH-EAST
		flag1 = isDuplicated(c, 1, -1);
		// SOUTH-WEST
		flag2 = isDuplicated(c, -1, 1);
		// SOUTH-EAST
		flag3 = isDuplicated(c, 1, 1);
		if(flag0 == true || flag1 == true || flag2 == true || flag3 == true){
			return true;
		}else{
			return false;
		}
	}
	
	private int decode(int code){
		int v = 0;
		switch(code){
		case 0: v =  0; break;
		case 1: v =  1; break;
		case 2: v =  2; break;
		case 6: v = -2; break;
		case 7: v = -1; break;
		default: break;
		}
		return v;
	}
	
	private void setup(int x, int y, int id, int rot){
		int d, v;
		check_len = lenTable.data[id];
		for(int i = 0; i < check_len; i++){
			// index = id * 5 * 8 + rot * 5 + i;
			d = table.data[(id << 5) + (id << 3) + (rot << 2) + rot + i];
			v = decode((d>>>3) & 0x07);
			check_x[i] = v + x;
			v = decode(d & 0x07);
			check_y[i] = v + y;
		}
	}
	
	public boolean isValid(int x, int y, int c, int id, int rot) {
		setup(x, y, id, rot);
		if (isInsideAndEmpty() == false) { return false; } // must be inside
		if (isEdgeSharing(c) == true) { return false; } // must not be shared edges with existing blocks
		if (isVertexSharing(c) == false) { return false; } // must be shared vertices with existing blocks
		return true;
	}
	
	public void move(int x, int y, int c, int id, int rot){
		int d, v;
		int mx, my;
		byte sym;
		check_len = lenTable.data[id];
		sym = symTable.data[id];
		for(int i = 0; i < check_len; i++){
			// index = id * 5 * 8 + rot * 5 + i;
			d = table.data[(id << 5) + (id << 3) + (rot << 2) + rot + i];
			v = decode((d>>>3) & 0x07);
			mx = v + x;
			v = decode(d & 0x07);
			my = v + y;
			put(c, mx, my, sym);
		}
	}
	
	
	public byte getPattern(int k, int r, int i){
		// index = k * 5 * 8 + r * 5 + i;
		return table.data[(k << 5) + (k << 3) + (r << 2) + r + i];
	}
	
	@unsynthesizable
	public static void main(String... args){
		BlokusBoard p = new BlokusBoard();
		
		for(int k = 0; k < 'u'-'a'+1; k++){
			System.out.printf("%c\n", k+'a');
			for(int r = 0; r < 8; r++){
				String s = "";
				for(int i = 0; i < 5; i++){
					byte b = p.getPattern(k, r, i);
					if((b & 0xff) != 0xff){
						int y = p.decode(b & 0x07);
						int x = p.decode((b >>> 3) & 0x07);
						s += " (" + x + ", " + y + ")";
					}
				}
				System.out.println("  " + s);
			}
		}
	}

}
