library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_signed.all;
 
entity upl_tx is
  generic (
    DATA_WIDTH : integer := 32;
    DATA_DEPTH : integer := 12
  );
  port (
    pReset_n : in std_logic;
    pUPLGlobalClk : in std_logic;
    
    -- UPLに出力する
    pO0Data    : out std_logic_vector(31 downto 0 );
    pO0Request : out std_logic;
    pO0Ack     : in  std_logic;
    pO0En      : out std_logic;
    
    pMemWriteData: in std_logic_vector(DATA_WIDTH-1 downto 0);
    pMemWriteAddr: in std_logic_vector(DATA_DEPTH-1 downto 0);
    pMemWriteEn  : in std_logic_vector(0 downto 0);

    pStart : in  std_logic;
    pBusy  : out std_logic;
    pDataLength : in std_logic_vector(31 downto 0);

    pSrcIPAddr  : in std_logic_vector(31 downto 0);
    pDestIPAddr : in std_logic_vector(31 downto 0);
    pSrcPort    : in std_logic_vector(15 downto 0);
    pDestPort   : in std_logic_vector(15 downto 0);

    pDebugA : out std_logic_vector( 7 downto 0 )
    );
end upl_tx;

architecture RTL of upl_tx is

  component simpledualportram
    generic (
      DEPTH : integer := 12;
      WIDTH : integer := 32);
    port (
      clk   : in  std_logic;
      we    : in  std_logic_vector(0 downto 0);
      raddr : in  std_logic_vector(DEPTH-1 downto 0);
      rdata : out std_logic_vector(WIDTH-1 downto 0);
      waddr : in  std_logic_vector(DEPTH-1 downto 0);
      wdata : in  std_logic_vector(WIDTH-1 downto 0)
      );
  end component;

  type UPLTxState is (IDLE, ACKWAIT, SEND_DATA);
  signal state : UPLTxState := IDLE;
  signal tx_counter : std_logic_vector(DATA_DEPTH-1 downto 0);

  signal cRAddr : std_logic_vector(DATA_DEPTH-1 downto 0);
  signal cRData : std_logic_vector(DATA_WIDTH-1 downto 0);

  signal cSrcIPAddr  : std_logic_vector(31 downto 0);
  signal cDestIPAddr : std_logic_vector(31 downto 0);
  signal cSrcPort    : std_logic_vector(15 downto 0);
  signal cDestPort   : std_logic_vector(15 downto 0);
  signal cDataLength : std_logic_vector(31 downto 0);

  signal cBusy : std_logic := '0';

begin  -- RTL

  M: simpledualportram
    generic map( DEPTH => DATA_DEPTH, WIDTH => DATA_WIDTH )
    port map(
      clk   => pUPLGlobalClk,
      we    => pMemWriteEn,
      raddr => cRAddr,
      rdata => cRData,
      waddr => pMemWriteAddr,
      wdata => pMemWriteData
      );
  
  pBusy <= pStart or cBusy;

  process (pUPLGlobalClk)
  begin  -- process
    if pUPLGlobalClk'event and pUPLGlobalClk = '1' then  -- rising clock edge
      if pReset_n = '0' then  -- asynchronous reset (active low)
        state        <= IDLE;
        cRAddr       <= (others => '0');
        pO0Request   <= '0';
        pO0En        <= '0';
        tx_counter   <= (others => '0');
        pDebugA <= "00000000";
        cBusy        <= '0';
      else
        case state is
          ---------------------------------------------------------------------------
          -- Javaからのリクエストを待つ
          ---------------------------------------------------------------------------
          when IDLE =>
--            if pUPLSendReq = '1' then
--              pUPLSendBusy <= '1';
--              pO0Request   <= '1';
--              state <= SEND;
--            else
--              pUPLSendBusy <= '0';
--            end if;
--            if pO0Ack = '0' and pStart = '1' then   -- Requestを発行していい状態
            if pStart = '1' then   -- Requestを発行していい状態
              pO0Request <= '1'; -- データの送信を要求
              state <= ACKWAIT;
              cSrcIPAddr  <= pSrcIPAddr;
              cDestIPAddr <= pDestIPAddr;
              cSrcPort    <= pSrcPort;
              cDestPort   <= pDestPort;
              cDataLength <= pDataLength;
              cBusy <= '1';
            else
              cBusy <= '0';
            end if;
            pO0En <= '0';
            tx_counter   <= (others => '0');
            cRAddr <= (others => '0');
            pDebugA <= "0000000" & pStart;
          ---------------------------------------------------------------------------
          -- UPLパケットを送れるまで待って送信を開始する
          ---------------------------------------------------------------------------
          when ACKWAIT =>
            if pO0Ack = '1' then        -- データを送れるようになった
              pO0Request <= '0';
              pO0En      <= '1';         -- データの送信開始
              pO0Data    <= cSrcIPAddr;  -- 送り元(自分)のIPアドレス
              tx_counter <= tx_counter + 1;
              state <= SEND_DATA;
            end if;
            pDebugA <= "00000010";
          ---------------------------------------------------------------------------
          -- UDPデータに相当するUPLパケットを送信する
          ---------------------------------------------------------------------------
          when SEND_DATA =>
            if (conv_integer(cDataLength) + 3 = tx_counter) then
              tx_counter <= (others => '0');
              state <= IDLE;
            else
              tx_counter <= tx_counter + 1;
            end if;
            case conv_integer(tx_counter) is
              when 0 => null;                              -- tx_counterが1に相当する送り元のIPアドレスは先のステートで送信済み
              when 1 => pO0Data <= cDestIPAddr;            -- 送り先のIPアドレス
              when 2 => pO0Data <= cSrcPort & cDestPort;   -- 送り元のポート番号
              when 3 =>
                pO0Data <= cDataLength(29 downto 0) & "00";     -- 送るデータの長さ(バイト数になおす)
                cRAddr  <= cRAddr + 1;  -- 次の次に供えてアドレスをインクリメントしておく
              when others =>            -- 実際にデータを送る
                pO0Data <= cRData;
                cRAddr  <= cRAddr + 1;
            end case;
            pDebugA <= "00000100";
          when others =>
            state <= IDLE;
        end case;
      end if;
    end if;
  end process;

end RTL;
