-------------------------------------------------------------------------------
-- M25P40 Buffer
--
--  written by Takefumi MIYOSHI (miyo@wasamon.net)
--  2013/4/6
-------------------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

entity m25p40_buf is
  
  port (
    clk   : in  std_logic;
    reset : in  std_logic;

    M25P40_C   : out std_logic;        -- Serial Clock
    M25P40_DQ0 : out std_logic;        -- Serial Data to M25P40
    M25P40_DQ1 : in std_logic;         -- Serial Data from M25P40
    M25P40_S : out std_logic;          -- Chip Select

    -- command
    pReq     : in  std_logic;                      -- do operation
    pBusy    : out std_logic;                      -- module running status
    pCommand : in  std_logic_vector(7 downto 0);   -- command
    pAddr    : in  std_logic_vector(23 downto 0);  -- read/write address
    pLength  : in  std_logic_vector(8 downto 0);   -- data length to read/write
    
    -- Dual-port RAM I/O
    data_waddr : in  std_logic_vector(7 downto 0);
    data_wdata : in  std_logic_vector(7 downto 0);
    data_we    : in  std_logic_vector(0 downto 0);
    data_raddr : in  std_logic_vector(7 downto 0);
    data_rdata : out std_logic_vector(7 downto 0);

    pDebug : out std_logic_vector(7 downto 0)
    );
  
end m25p40_buf;

architecture RTL of m25p40_buf is

  constant WRITE_ENABLE                    : std_logic_vector(7 downto 0) := "00000110"; -- 06h
  constant WRITE_DISABLE                   : std_logic_vector(7 downto 0) := "00000100"; -- 04h
  constant READ_IDENTIFICATION             : std_logic_vector(7 downto 0) := "10011111"; -- 9Fh
  constant READ_STATUS_REGISTER            : std_logic_vector(7 downto 0) := "00000101"; -- 05h
  constant WRITE_STATUS_REGISTER           : std_logic_vector(7 downto 0) := "00000001"; -- 01h
  constant READ_DATA_BYTES                 : std_logic_vector(7 downto 0) := "00000011"; -- 03h
  constant READ_DATA_BYTES_AT_HIGHER_SPEED : std_logic_vector(7 downto 0) := "00001011"; -- 0Bh
  constant PAGE_PROGRAM                    : std_logic_vector(7 downto 0) := "00000010"; -- 02h
  constant SECTOR_ERASE                    : std_logic_vector(7 downto 0) := "11011000"; -- D8h
  constant BULK_ERASE                      : std_logic_vector(7 downto 0) := "11000111"; -- C7h
  constant DEEP_POWER_DOWN                 : std_logic_vector(7 downto 0) := "10111001"; -- B9h
  constant RELEASE_FROM_DEEP_POWER_DOWN    : std_logic_vector(7 downto 0) := "10101011"; -- ABh

  signal busy       : std_logic := '0';
  signal serial_clk : std_logic := '0';

  signal command    : std_logic_vector(7 downto 0)  := (others => '0');
  signal addr       : std_logic_vector(23 downto 0) := (others => '0');
  signal length     : std_logic_vector(8 downto 0)  := (others => '0');
  signal buf : std_logic_vector(7 downto 0) := (others => '0');

  type STATE_TYPE is (IDLE, EMIT, EMIT_COMMAND, EMIT_ADDR0, EMIT_ADDR1, EMIT_ADDR2, SEND_DATA, RECV_DATA, OP_DONE);
  signal state : STATE_TYPE := IDLE;
  signal next_state : STATE_TYPE := IDLE;

  signal counter : std_logic_vector(7 downto 0) := (others => '0');
  signal subcounter : std_logic_vector(31 downto 0) := (others => '0');

  signal recv_counter, send_counter : std_logic_vector(8 downto 0);

  signal ram_read_addr, ram_write_addr : std_logic_vector(7 downto 0);
  signal ram_read_data, ram_write_data : std_logic_vector(7 downto 0);
  signal ram_write_en      : std_logic_vector(0 downto 0);

  component simpledualportram
    generic (
      DEPTH : integer := 10;
      WIDTH : integer := 32);
    port (
      clk   : in  std_logic;
      we    : in  std_logic_vector(0 downto 0);
      raddr : in  std_logic_vector(DEPTH-1 downto 0);
      rdata : out std_logic_vector(WIDTH-1 downto 0);
      waddr : in  std_logic_vector(DEPTH-1 downto 0);
      wdata : in  std_logic_vector(WIDTH-1 downto 0)
      );
  end component;

begin  -- RTL

  pBusy     <= busy or pReq;
  M25P40_C <= serial_clk;

  U_SEND_BUF: simpledualportram
    generic map(DEPTH => 8, WIDTH => 8)
    port map(
      clk   => clk,
      we    => data_we,
      raddr => ram_read_addr,
      rdata => ram_read_data,
      waddr => data_waddr,
      wdata => data_wdata
      );

  U_RECV_BUF: simpledualportram
    generic map(DEPTH => 8, WIDTH => 8)
    port map(
      clk   => clk,
      we    => ram_write_en,
      raddr => data_raddr,
      rdata => data_rdata,
      waddr => ram_write_addr,
      wdata => ram_write_data
      );

  process (clk)
  begin  -- process
    if clk'event and clk = '1' then   -- rising clock edge
      if reset = '1' then
        M25P40_DQ0   <= '0';
        M25P40_S     <= '1';
        pDebug       <= (others => '0');
        busy         <= '0';
        command      <= (others => '0');
        addr         <= (others => '0');
        serial_clk   <= '0';
        state        <= IDLE;
        next_state   <= IDLE;
        counter      <= (others => '0');
        subcounter   <= (others => '0');
        buf          <= (others => '0');
        ram_write_en <= "0";
      else
        case state is

          when IDLE =>
            M25P40_DQ0   <= '0';
            pDebug       <= (others => '0');
            serial_clk   <= '0';
            counter      <= (others => '0');
            subcounter   <= (others => '0');
            recv_counter <= (others => '0');
            send_counter <= (others => '0');
            ram_write_en <= "0";
            if pReq = '1' then
              busy     <= '1';
              addr     <= pAddr;
              command  <= pCommand;
              length   <= pLength;
              M25P40_S <= '0';
              state    <= EMIT_COMMAND;
            else
              busy     <= '0';
              M25P40_S <= '1';
            end if;

          when EMIT_COMMAND =>
            case command is
              when WRITE_ENABLE =>
                next_state <= OP_DONE;
              when WRITE_DISABLE =>
                next_state <= OP_DONE;
              when READ_IDENTIFICATION =>
                next_state   <= RECV_DATA;
                length       <= conv_std_logic_vector(1 + 2 + 17, length'length);
                recv_counter <= (others => '0');
              when READ_STATUS_REGISTER =>
                next_state   <= RECV_DATA;
                length       <= conv_std_logic_vector(1, length'length);
                recv_counter <= (others => '0');
              when WRITE_STATUS_REGISTER =>
                next_state   <= SEND_DATA;
                send_counter <= (others => '0');
                length       <= conv_std_logic_vector(1, length'length);
              when READ_DATA_BYTES =>
                next_state <= EMIT_ADDR0;
              when READ_DATA_BYTES_AT_HIGHER_SPEED =>  -- not supported
                next_state <= OP_DONE;
              when PAGE_PROGRAM =>
                next_state <= EMIT_ADDR0;
              when SECTOR_ERASE =>
                next_state <= EMIT_ADDR0;
              when BULK_ERASE =>
                next_state <= OP_DONE;
              when DEEP_POWER_DOWN =>                  -- not supported
                next_state <= OP_DONE;
              when RELEASE_FROM_DEEP_POWER_DOWN =>     -- not supported
                next_state <= OP_DONE;
              when others =>
                next_state <= OP_DONE;
            end case;
            state      <= EMIT;
            buf        <= command;
            subcounter <= (others => '0');
            counter    <= (others => '0');
            
          when EMIT_ADDR0 =>
            next_state <= EMIT_ADDR1;
            state      <= EMIT;
            buf        <= addr(23 downto 16);
            subcounter <= (others => '0');
            counter    <= (others => '0');
          when EMIT_ADDR1 =>
            next_state <= EMIT_ADDR2;
            state      <= EMIT;
            buf        <= addr(15 downto 8);
            subcounter <= (others => '0');
            counter    <= (others => '0');
          when EMIT_ADDR2 =>
            case command is
              when READ_DATA_BYTES =>
                next_state   <= RECV_DATA;
                recv_counter <= (others => '0');
              when PAGE_PROGRAM =>
                next_state   <= SEND_DATA;
                send_counter <= (others => '0');
              when SECTOR_ERASE =>
                next_state <= OP_DONE;
              when others =>
                next_state <= OP_DONE;
            end case;
            state      <= EMIT;
            buf        <= addr(7 downto 0);
            subcounter <= (others => '0');
            counter    <= (others => '0');
            
          when EMIT =>
            case conv_integer(subcounter) is
              when 0 =>
                counter <= counter + 1;
                M25P40_DQ0 <= buf(7);
                subcounter <= subcounter + 1;
              when 1 =>
                serial_clk <= '1';
                subcounter <= subcounter + 1;
              when 2 =>
                serial_clk <= '0';
                subcounter <= subcounter + 1;
              when 3 =>
                subcounter <= (others => '0');
                if conv_integer(counter) = 8 then
                  state      <= next_state;
                  counter    <= (others => '0');
                else
                  buf <= buf(6 downto 0) & '0';
                end if;
              when others =>
                state <= OP_DONE;  
            end case;
            
          --------------------------------------------------------------
          -- Store the data received from ROM into the internal Block RAM
          --------------------------------------------------------------
          when RECV_DATA =>
            if conv_integer(recv_counter) = conv_integer(length) then
              state        <= OP_DONE;
              ram_write_en <= "0";
            else
              case conv_integer(subcounter) is
                when 0 =>
                  counter      <= counter + 1;
                  subcounter   <= subcounter + 1;
                  serial_clk   <= '1';
                  ram_write_en <= "0";
                when 1 =>
                  buf          <= buf(6 downto 0) & M25P40_DQ1;
                  subcounter   <= subcounter + 1;
                  ram_write_en <= "0";
                when 2 =>
                  serial_clk   <= '0';
                  subcounter   <= subcounter + 1;
                  ram_write_en <= "0";
                when 3 =>
                  subcounter <= (others => '0');
                  if conv_integer(counter) = 8 then
                    ram_write_addr     <= recv_counter(7 downto 0);
                    ram_write_en <= "1";
                    ram_write_data <= buf;
                    recv_counter <= recv_counter + 1;
                    counter      <= (others => '0');
                  else
                    ram_write_en <= "0";
                  end if;
                when others =>
                  ram_write_en <= "0";
                  state        <= OP_DONE;
              end case;
            end if;
            
          --------------------------------------------------------------
          -- send the data stored in the internal Block RAM to ROM
          --------------------------------------------------------------
          when SEND_DATA =>
            case conv_integer(subcounter) is
              when 0 =>
                if conv_integer(send_counter) = conv_integer(length) then
                  case command is
                    when WRITE_STATUS_REGISTER => state <= OP_DONE;
                    when PAGE_PROGRAM          => state <= OP_DONE;
                    when others                => state <= OP_DONE;
                  end case;
                else
                  ram_read_addr <= send_counter(7 downto 0);
                  send_counter  <= send_counter + 1;
                  subcounter    <= subcounter + 1;
                end if;
              when 1 =>
                subcounter <= subcounter + 1;
              when 2 =>
                buf        <= ram_read_data;
                state      <= EMIT;
                next_state <= SEND_DATA;  -- in order to back to this state
                subcounter <= (others => '0');
              when others =>
                state <= OP_DONE;
            end case;
            
          when OP_DONE =>
            M25P40_S  <= '1';
            state <= IDLE;

          when others =>
            state <= OP_DONE;
        end case;
      end if;
    end if;
  end process;

  

end RTL;
